c***********************************************************************
! UMATHT & UMAT subroutines for the phase field model for fracture
! Coupled diffusion-elastoplasticity version  
c***********************************************************************
! If using this code for research or industrial purposes please cite:
! L. Castro, Y. Navidtehrani, C. Betegon, E. Martinez-Paneda. 
! Coupled thermo-chemo-mechanical phase field-based modelling of 
! hydrogen-assisted cracking in girth welds. 
! International Journal of Mechanical Sciences 311, 111172 (2026)
! 
! This code is provided “as is”, without any warranties or guarantees.
! All rights reserved
c*****************************************************************
! Transfering variables between subroutines and Parts
      module ktransfer
        integer,  parameter :: ninpt = 4  !number of integration points (CPS4/CPS8R/CEP4/CPE8R/C3D10=4; C3D8/C3D20R=8)
        integer,  parameter :: ns = 4     !number of stress components (CPS4/CPS8R=3; CPE4/CPE8R=4; C3D8/C3D10/C3D20R=6)
        integer,  parameter :: nnod = 8   !number of nodes per element (CPS8R/CPE8R=8)
        integer,  parameter :: ntraps = 4 !number of traps  UPDATE
		integer,  parameter :: kflagHydrogenTraps = 1 ! =0 No traps / =1 Traps
        integer,  parameter :: nelemMax = 1000000 !max number of total elements possible
        integer,  parameter :: kflagHard = 1                !flag to consider power law hardening (0 = linear; 1 = power law)
        integer,  parameter :: kflagAdaptiveTimeUMAT = 1    !flag to use adative time step in umat in case of plasticity algo not converging 1/0 (yes/no)
        integer,  parameter :: kflagComputeLenghscale = 1   !flag to compute length scale for AT1/2 1/0 (yes/no)
        integer,  parameter :: kflagHydrogenLenghscale = 0  !flag to compute length scale using degraded Gc for AT1/2 1/0 (yes/no)
        integer,  parameter :: kflagPenaltyCPhi =1         !flag to consider penalty method based implementation of hydrogen BC in damaged region
        integer,  parameter :: kflagPenaltyDPhi = 1         !flag to consider diffusivity based moving hydrogen BC in damaged region 
		integer,  parameter :: kflagPenaltyIncr = 0         !incremental hydrogen BC
        real*8 :: C_env = 0.2434953798d0  !Hydrogen concentration in environment 10 MPa Pressure(wppm) !Update
        real*8 :: pi = 3.141592d0	
        real*8 :: conc_factor = 2.1368e-16 !1 atom H/mm^3 = 2.1368e-16 ppmw 		
        real*8 :: phi_th = 0.9d0 ! Threshold on damage for moving Hydrogen BC
        !!Transfer variables between parts 
        real*8 :: TransPhi(nelemMax,ninpt) = 0.d0
	    real*8 :: TranscL(nelemMax,ninpt)= 0.d0
		real*8 :: Trans_peeq(nelemMax,ninpt) = 0.d0
	    real*8 :: xjaci(2,2) = 0.d0
	    real*8 :: xjacm(2,2) = 0.d0
	    real*8 :: djacb = 0.d0
	    real*8 :: Transrho(nelemMax,ninpt) = 0.d0
		real*8 :: Transrho_ssd(nelemMax,ninpt) = 0.d0
	    real*8 :: ShT(nelemMax,ninpt) = 0.d0
	    real*8 :: coorT(nelemMax,ninpt,2) = 0.d0
	    real*8 :: deriv(2,4) = 0.d0
	    real*8 :: grad(nelemMax,nnod) = 0.d0
		real*8 :: XINstrain(4, 4, nelemMax)=0.d0
	    integer :: KflagINIT(4, nelemMax)  
	    real*8 :: usr_tolerance = 1e-20 !General user tolerance 
	      save
      end module  

!***********************************************************************
! Umat subroutine for mechanical and phase field DOFs	  
!***********************************************************************
       subroutine umat(stress,statev,ddsdde,sse,spd,scd,rpl,ddsddt,
     1 drplde,drpldt,stran,dstran,time,dtime,temp,dtemp,predef,dpred,
     2 cmname,ndi,nshr,ntens,nstatv,props,nprops,coords,drot,pnewdt,
     3 celent,dfgrd0,dfgrd1,noel,npt,layer,kspt,kstep,kinc)
c
 
	  use ktransfer
      include 'aba_param.inc' !implicit real(a-h o-z)
c
      character*80 cmname
	  character*80 cpname
	  
      dimension stress(ntens), statev(nstatv),
     1 ddsdde(ntens,ntens),ddsddt(ntens),drplde(ntens),
     2 stran(ntens),dstran(ntens),time(2),predef(1),dpred(1),
     3 props(nprops),coords(3),drot(3,3),
     4 dfgrd0(3,3),dfgrd1(3,3)
c
      dimension eelas(ntens), eplas(ntens), flow(ntens), ps(ndi),
     1 stressOld(ntens), eelas_dev(ntens), stress0(ntens)

      parameter (one=1.0d0, two=2.0d0, three=3.0d0, six=6.0d0)
      parameter (newton = 100, toler = 1.d-8)
      
	  
!!!!!! Part 1 including mechanical and phase field DOFs	  
      if (cmname.eq.'MATERIAL-1') then

       call getpartinfo(noel, 1, cpname, locnum, jrcd)
	   
      ! adaptive time stepping
       pnewdtLocal = pnewdt
	  

      ! Store the initial elastic strain 
      if (KflagINIT(npt,noel) .ne. 1) then
        XINstrain(1:ntens,npt,noel) = statev(ntens + 1 : 2*ntens) 
        KflagINIT(npt,noel)=1
      endif

      ! Update the incremental strain
      if(kstep.eq.1 .and. kinc.eq.1) then
       dstran = dstran + XINstrain(1:ntens,npt,noel)
       statev(ntens + 1 : 2*ntens) = 0.d0 
      endif

c -----------------------------------------------------------
c     umat for isotropic elasticity and isotropic plasticity
c     j2 flow theory
c     can not be used for plane stress
c -----------------------------------------------------------
c     props(1) - e
c     props(2) - nu
c     props(3) - wtSpd
c     props(4) - kflagSplit
c     props(5) - kflagPFM
c     props(6) - ft
c     props(7) - syield
c     calls ahard for curve of syield vs. peeq
c -----------------------------------------------------------

!     Mechanical parameters
      emod=props(1) ! Young modulus 
	  emod0 = emod ! Keep the undegraded young's modulus
      enu=props(2)  ! Poisson's ratio
      wtSpd = props(3) ! Wt for plastic strain energy
      kflagSplit = props(4) ! Flag for split of strain energy =1/0 = Yes(Deviatoric-volumetric split)/No
      kflagPFM = props(5) ! Flag for PFM (0=AT2, 1=AT1, 2=PF-CZM)
	  ft = props(6) ! Energy for crack initiation
	  sy0 = props(7) ! Initial yield strength
     
      ! Hmin
      if(kflagPFM.eq.2) then
        Hmin = 0.5d0*ft**2.d0/emod0
      endif

      ! Phase field DOF
       phi = temp + dtemp ! latest phase field damage
     
      call phasefieldModel(kflagPFM, xlc, Gc, xi, c0, Hmin)
      call phasefieldDegFunc (kflagPFM, phi, xlc, Gc, Hmin, omega)
      
      ! Compute plasticity based on damaged stress 
      omegaE = omega ! degradation func for elasticity
      omegaP = 1.d0 + (omega - 1.d0)*wtSpd ! degradation func for plasticity
      emod = emod * omegaE
      
      ! Elastic parameters
      if(enu.gt.0.4999.and.enu.lt.0.5001) enu=0.499
      ebulk3=emod/(one-two*enu)
      ebulk = ebulk3 / three
      eg2=emod/(one+enu)
      eg=eg2/two
      eg3=three*eg
      elam=(ebulk3-eg2)/three

!     Elastic stiffness
      ddsdde = 0.d0
      do 40 k1=1,ndi
        do 30 k2=1,ndi
           ddsdde(k2,k1)=elam
 30     continue
        ddsdde(k1,k1)=eg2+elam
 40   continue
      do 50 k1=ndi+1,ntens
        ddsdde(k1,k1)=eg
 50   continue

!    Recover elastic, plastic strains, stress, PEEQ, energy densities
      stressOld = statev(1:ntens)
      eelas = statev(ntens + 1: 2*ntens) + dstran
      eplas = statev(2*ntens + 1: 3*ntens)
      eqplas = statev(3*ntens + 1)
      H_sse_pos0 = statev(3*ntens + 2)
      spd0 = statev(3*ntens + 3)
      
	  
!    Calculate trial stress from elastic strains (elastic prediction)
      stress = matmul(ddsdde, eelas)

!    If no yield stress is given, material is taken to be elastic

      if( (nprops.gt.7) .and. (props(7).gt.0.0) ) then

!       Mises stress
        smises=(stress(1)-stress(2))**two +
     1          (stress(2)-stress(3))**two +
     1          (stress(3)-stress(1))**two
        do 90 k1=ndi+1,ntens
              smises=smises+six*stress(k1)**two
 90     continue
        smises=sqrt(smises/two)

!       Hardening curve, get yield stress
        nvalue=(nprops - 6)/2 ! update it as per entries of props
        call ahard(syiel0,hard,eqplas,props(7),nvalue,noel,coords,temp,emod0)
        syiel0 = syiel0 * omegaP
        hard = hard * omegaP

!       Determine if actively yielding

        if (smises.gt.(1.0+toler)*syiel0) then

!         Flow direction

          shydro=(stress(1)+stress(2)+stress(3))/three
          onesy=one/smises
          do 110 k1=1,ndi
             flow(k1)=onesy*(stress(k1)-shydro)
 110      continue
          do 120 k1=ndi+1,ntens
             flow(k1)=stress(k1)*onesy
 120      continue

!       Solve for equiv stress, newton iteration

          syield=syiel0
          deqpl=0.0
          do 130 kewton=1,newton
             rhs  = ( smises - eg3*deqpl-syield )
             deqpl = deqpl + rhs / (eg3+hard)
             call ahard(syield,hard,eqplas+deqpl,props(7),nvalue,noel,coords,temp,emod0)
             syield = syield * omegaP
             hard = hard * omegaP
             if(abs(rhs).lt.toler*syiel0) goto 140
 130      continue

c         write only first few warning msg
          if(numWrite.lt.100) then
           write(7,2) newton, noel
 2         format('plasticity algorithm did not converge after ', i3,' iterations', ' in element', i10)
           numWrite = numWrite + 1
          endif
          if(kflagAdaptiveTimeUMAT) pnewdt = 0.1d0 ! adaptive time step
 140      continue


!       Calculate stress and update strains::plastic correction

          do 150 k1=1,ndi
             stress(k1)=flow(k1)*syield+shydro
             eplas(k1)=eplas(k1)+three*flow(k1)*deqpl/two
             eelas(k1)=eelas(k1)-three*flow(k1)*deqpl/two
 150      continue
          do 160 k1=ndi+1,ntens
             stress(k1)=flow(k1)*syield
             eplas(k1)=eplas(k1)+three*flow(k1)*deqpl
             eelas(k1)=eelas(k1)-three*flow(k1)*deqpl
 160      continue
          eqplas=eqplas+deqpl

!         Plastic strain energy density
           spd0 = spd0 + deqpl*(syiel0+syield)/two / omegaP
           spd = omegaP * spd0 
          
!       Calculate algorithmic consistent tangent

          effg=eg*syield/smises
          effg2=two*effg
          effg3=three*effg2/two
          efflam=(ebulk3-effg2)/three
          effhrd=eg3*hard/(eg3+hard) - effg3
          do 220 k1=1,ndi
             do 210 k2=1,ndi
                ddsdde(k2,k1)=efflam
 210         continue
             ddsdde(k1,k1)=effg2+efflam
 220      continue
          do 230 k1=ndi+1,ntens
             ddsdde(k1,k1)=effg
 230      continue
          do 250 k1=1,ntens
             do 240 k2=1,ntens
                ddsdde(k2,k1)=ddsdde(k2,k1)+flow(k2)*flow(k1)*effhrd
 240         continue
 250      continue
        endif
      endif

!     !Plastic strain energy density for crack driving
!     (sse = sigma:ee = sse + sigma : delta_ee) 
!     Deviatoric/volumetric split (Amor et al)
      if (kflagSplit.eq.1) then

       ! - Decompose elastic strain tensor
       eelas_tr = eelas(1) + eelas(2) + eelas(3)
       eelas_dev(1:3) = eelas(1:3) - eelas_tr / three
       eelas_dev(4:ntens) = eelas(4:ntens)

       eelas_dev_sq = 0.0d0
       do k1=1,ntens
          if(k1.le.ndi) then 
            eelas_dev_sq = eelas_dev_sq + eelas_dev(k1)**2.d0
          else
            eelas_dev_sq = eelas_dev_sq + 0.5d0 * eelas_dev(k1)**2.d0
          endif
       enddo

       eelas_tr_pos = (eelas_tr + abs(eelas_tr)) / two ! <tr(EE)>
       eelas_tr_neg = (eelas_tr - abs(eelas_tr)) / two
       sse_pos = 0.5d0 * ebulk * eelas_tr_pos ** two + eg * eelas_dev_sq
       sse_neg = 0.5d0 * ebulk * eelas_tr_neg ** two

!     Principal strain based (Miehe et al)
      elseif (kflagSplit.eq.2) then
       call sprinc(eelas, ps, 2, ndi, nshr) ! compute principal elastic strain
       ps_tr = ps(1)+ps(2)+ps(3)
       trp1=(ps_tr + abs(ps_tr))/2.d0
       trn1=(ps_tr - abs(ps_tr))/2.d0
       trp2=0.d0; trn2=0.d0
       do i=1,3
          trp2=trp2+(ps(i)+abs(ps(i)))**2.d0/4.d0
          trn2=trn2+(ps(i)-abs(ps(i)))**2.d0/4.d0
       end do
       sse_pos = enu*eg/(1.d0-2.d0*enu)*trp1**2.d0+eg*trp2
       sse_neg = enu*eg/(1.d0-2.d0*enu)*trn1**2.d0+eg*trn2

c     no split
      else
       sse_pos = 0.5d0 * dot_product (stress, eelas)
       sse_neg = 0.d0
      endif


!     Update elastic strain energy density 
      sse = sse_pos + sse_neg ! elastic strain energy density (damaged)
      sse_pos0 = sse_pos / omegaE ! undamaged positive strain energy density

      ! Nominal hydrostatic stress
      SH = (stress(1) + stress(2) + stress(3))/3.d0

      ! Crack driving energy
      H_sse_pos0 = max(H_sse_pos0, sse_pos0)
      se_driving = (wtSpd * spd0 + H_sse_pos0) 
      if(kflagPFM.eq.0) then 
        Hmin0 = 0.d0
      else    
        Hmin0 = Hmin
      endif
	  
!     Driving Force
      H = max( se_driving, Hmin0)

	  ! Strain creep dissipation
	  scd = (wtSpd * spd0 + H_sse_pos0)
      
!     Store information for computing the Sh gradient     
      ShT(noel,npt)=(stress(1)+stress(2)+stress(3))/3.d0      
      coorT(noel,npt,1)=coords(1)
      coorT(noel,npt,2)=coords(2)	  
!     Compute the gradient of the hydrostatic stress    
      if (npt==1 .and. time(1).gt.0) then
       do k1=1,4 ! Hard coded for 4 integration points
        if (k1==1) then
         s=-1.d0
         t=-1.d0
        elseif (k1==2) then 
         s=1.d0
         t=-1.d0
        elseif (k1==3) then
         s=-1.d0
         t=1.d0  
        elseif (k1==4) then      
         s=1.d0
         t=1.d0   
        end if

        deriv(1,1)=-(1.d0/4.0)*(1-t)
        deriv(1,2)=(1.d0/4.0)*(1-t)
        deriv(1,3)=-(1.d0/4.0)*(1+t)
        deriv(1,4)=(1.d0/4.0)*(1+t)
        deriv(2,1)=-(1.d0/4.0)*(1-s)
        deriv(2,2)=-(1.d0/4.0)*(1+s)
        deriv(2,3)=(1.d0/4.0)*(1-s)
        deriv(2,4)=(1.d0/4.0)*(1+s)

        xjacm(1,1)=deriv(1,1)*coorT(noel,1,1)+deriv(1,2)*coorT(noel,2,1)
     1 +deriv(1,3)*coorT(noel,3,1)+deriv(1,4)*coorT(noel,4,1)
    
        xjacm(1,2)=deriv(1,1)*coorT(noel,1,2)+deriv(1,2)*coorT(noel,2,2)
     1 +deriv(1,3)*coorT(noel,3,2)+deriv(1,4)*coorT(noel,4,2)
     
        xjacm(2,1)=deriv(2,1)*coorT(noel,1,1)+deriv(2,2)*coorT(noel,2,1)
     1 +deriv(2,3)*coorT(noel,3,1)+deriv(2,4)*coorT(noel,4,1)
      
        xjacm(2,2)=deriv(2,1)*coorT(noel,1,2)+deriv(2,2)*coorT(noel,2,2)
     1 +deriv(2,3)*coorT(noel,3,2)+deriv(2,4)*coorT(noel,4,2)

        djacb=xjacm(1,1)*xjacm(2,2)-xjacm(1,2)*xjacm(2,1) 
      
        xjaci(1,1)=xjacm(2,2)/djacb 
        xjaci(1,2)=-xjacm(1,2)/djacb  
        xjaci(2,1)=-xjacm(2,1)/djacb   
        xjaci(2,2)=xjacm(1,1)/djacb

        a1=xjaci(1,1)*deriv(1,1)+xjaci(1,2)*deriv(2,1) 
        a2=xjaci(1,1)*deriv(1,2)+xjaci(1,2)*deriv(2,2) 
        a3=xjaci(1,1)*deriv(1,3)+xjaci(1,2)*deriv(2,3) 
        a4=xjaci(1,1)*deriv(1,4)+xjaci(1,2)*deriv(2,4) 
        b1=xjaci(2,1)*deriv(1,1)+xjaci(2,2)*deriv(2,1) 
        b2=xjaci(2,1)*deriv(1,2)+xjaci(2,2)*deriv(2,2) 
        b3=xjaci(2,1)*deriv(1,3)+xjaci(2,2)*deriv(2,3) 
        b4=xjaci(2,1)*deriv(1,4)+xjaci(2,2)*deriv(2,4)  
      
        grad(noel,2*k1-1)=a1*ShT(noel,1)+a2*ShT(noel,2)+a3*ShT(noel,3)
     1 +a4*ShT(noel,4)
        grad(noel,2*k1)=b1*ShT(noel,1)+b2*ShT(noel,2)+b3*ShT(noel,3)
     1 +b4*ShT(noel,4)
       end do 
      end if 
      !Transfer cL from Part-2 for visualization
      cL=TranscL(locnum,npt)
! Store stresses and strains in state variable array
      statev(1:ntens) = stress
      statev(ntens+1:2*ntens) = eelas
      statev(2*ntens+1:3*ntens) = eplas
      statev(3*ntens+1) = eqplas ! cum. eqv. plastic strain PEEQ
	  Trans_peeq(locnum,npt) = eqplas 
      statev(3*ntens+2) = H_sse_pos0 ! history variable for tensile elastic strain energy density
      statev(3*ntens+3) = spd0 ! plastic strain energy density (undamaged)
      statev(3*ntens+4) = phi ! phase field damage
      statev(3*ntens+5) = H ! crack driving energy
      statev(3*ntens+6) = cL ! lattice hydrogen concentration cL [wt ppmw]
      statev(3*ntens+9) = syiel0 ! yield strength (function of computed PEEQ)
      statev(3*ntens+10) = grad(noel,2*npt-1) ! gradient of hydrostatic stress x direction
	  statev(3*ntens+11) = grad(noel,2*npt) ! gradient of hydrostatic stress y direction
      statev(3*ntens+12) = emod ! young's modulus
	  
	  
!!!!!! Part 2 including hydrogen concentration DOF
	  elseif(cmname.eq.'MATERIAL-2') then
      
       stress=0.d0
       ddsdde=0.d0
	   
      end if
	   
      end subroutine umat


!***********************************************************************
!    Umatht to define Phase Field and Hydrogen diffusion DOFs
!***********************************************************************
      subroutine umatht(u,dudt,dudg,flux,dfdt,dfdg,
     1 statev,temp,dtemp,dtemdx,time,dtime,predef,dpred,
     2 cmname,ntgrd,nstatv,props,nprops,coords,pnewdt,
     3 noel,npt,layer,kspt,kstep,kinc)
c
	  use ktransfer
      include 'aba_param.inc'
c
      character*80 cmname
	  character*80 cpname
	  
      dimension dudg(ntgrd),flux(ntgrd),dfdt(ntgrd),
     1 dfdg(ntgrd,ntgrd),statev(nstatv),dtemdx(ntgrd),
     2 time(2),predef(1),dpred(1),props(nprops),coords(3)
      dimension Wb(ntraps),xK(ntraps),xNt(ntraps),sig(ntgrd), cT(ntraps),thetaT(ntraps)
	  
!!!!!! Part 1 including mechanical and phase field DOFs
      if (cmname.eq.'MATERIAL-1') then
	    call getpartinfo(noel, 1, cpname, locnum, jrcd)
		emod0=props(1) ! Young modulus 
		Gc0=props(2) ! Fracture energy
		Gcmin=props(3) ! Min Fracture energy Following law
		ft=props(4) ! Energy for crack initiation
		ft0=ft
	    ! Phase field variable
	    phi=temp+dtemp 
		! Update driving force
	    H=statev(17)
	    ! Transfer Hydrogen Concentation
	    cL = TranscL(locnum,npt)
	    if(cL.gt.usr_tolerance) then
	    ! Parameters of Degradation Law
		 xm_ = 9.0d0; xn_ = 0.8d0;
	    ! Hydrogen dependent degradation 
	    Gc = Gcmin + (Gc0 - Gcmin)*exp(-xm_*cL**xn_)
	    else
	    Gc = Gc0
	    endif
	    fc_hydrogen = Gc / Gc0
	    fc_hydrogen= max(fc_hydrogen, usr_tolerance)
	    Gc = fc_hydrogen * Gc0
      
       ! Compute lengthscale from {E,Gc,ft}
       if(kflagComputeLenghscale)then
       if(kflagHydrogenLenghscale) then
        xlch = emod0 * Gc / ft**2.d0 !use degraded Gc=> length scale changes with hydrogen
       else
        xlch = emod0 * Gc0 / ft0**2.d0 !length scale independent of hydrogen
       endif
       if(kflagPFM.eq.0) then
       xlc = 27.0d0/256.0d0 * xlch
       elseif(kflagPFM.eq.1) then 
       xlc = 3.0d0/8.0d0 * xlch
       endif     
       endif
	 
       U=U+(phi/xlc**2-2.d0*(1.d0-phi)*H/(Gc*xlc))*DTIME ! Internal energy
       DUDT=(1.d0/xlc**2+2.d0*H/(Gc*xlc))*DTIME ! Derivative of internal energy w.r.t. time
       DUDG=0.d0 ! Derivative of internal energy w.r.t. gradient of phase field
       DFDT=0.d0 ! Derivative of flux w.r.t. phase field
       do i=1,NTGRD
        DFDG(i,i)=-1.d0 ! Derivative of flux w.r.t. gradient of phase field, 
       end do
       FLUX=matmul(DFDG,DTEMDX) ! Flux
       TransPhi(noel,npt)=phi
	   ! Store SDV
	  statev(19) = xlc ! xlc [mm]
      statev(20) = Gc ! Gc(C) [N/mm]
!!!!!! Part 2 including hydrogen concentration DOF
      elseif(cmname.eq.'MATERIAL-2') then
      call getpartinfo(noel, 1, cpname, locnum, jrcd)
	  
	  ! Parameters of UMATHT
	  dfdg=0.d0
	  du2=0.d0
      ntens=ntgrd*2

      ! Lattice Diffusion Coefficient
      D=props(1)
	  
	  ! Penalty method to assign hydrogen BC in the damaged regime
      pen_rhs = 0.d0; pen_mat = 0.d0;pen_fac=0.d0;pen_D=0.0d0
      if(kflagPenaltyCPhi) then 
       phi=TransPhi(locnum,npt) ! phase field from umat
       pen_wt = max(phi-phi_th, 0.d0)
      ! Diffusivity based moving BC
      if(kflagPenaltyDPhi) then
       pen_fac = 1.0d4 ! penalty factor
	   pen_D = 0.0d0
       D = D * (1.0d0 + pen_wt * pen_fac * dtime)
      else
	  ! Penalty Method
       pen_fac = 1.0d4 ! penalty factor
       pen_rhs = pen_fac*pen_wt*(temp - C_env)
       pen_mat = pen_fac*pen_wt
	   pen_D = 1.0d0
      endif
      endif
	  
      ! Constants to solve hydrogen diffusion equation
      Vh=2000.d0 ! [mm^3/mol]
      R=8314.5d0 ! [N*mm/(mol*K)]
      T=300.d0   ! [K]
	  
	  ! Import gradient of damaged stress from UMAT (Interpolation method)
      sig(1)=grad(locnum,2*npt-1)
      sig(2)=grad(locnum,2*npt)
	 
	  ! Binding energies of each Trap
       Wb(1) = 25.0d0        ! Dislocation binding energy (kJ/mol)
       Wb(2) = 47.1d0        ! Martensitic-austenitic interfaces (kJ/mol)
       Wb(3) = 13.5d0        ! Cementite-ferrite interfaces (kJ/mol)
       Wb(4) = 32.0d0        ! Grain Bonudaries (kJ/mol)
	   ! Parameters
	   xNl = 5.2d20 ! Number of lattice sites per unit volume [1/mm^3] [sites/mm^3] Fernandez-Sousa et al., Acta Materialia, 2020.
	   
       ! Equilibrium constants
	   do k1=1,ntraps
	   xK(k1) = exp(Wb(k1)*1e6/(R*T))
	   end do
	   
	  ! Traps density 
	   if (kflagHydrogenTraps.eq.0) then ! Lattice H only
       xNt(1)=0.d0
	   xNt(2)=0.d0
	   xNt(3)=0.d0
	   xNt(4)=0.d0
	   elseif (kflagHydrogenTraps.eq.1) then ! kflag=1 Traps
	   ! Transfer PlasticStrain
	   eplasequi=Trans_peeq(locnum,npt)
	   d_lattice=2.86e-7 ! lattice parameter(mm)
	   ! Traps Density
	   if (eplasequi.le.0.5) then
          xNt(1)=sqrt(2.d0)*(1e10+eplasequi*2e16)/(d_lattice*1e6) !(sites/mm^3)
          du2=(xK(1)*temp/(xK(1)*temp/conc_factor+xNl))*(sqrt(2.d0)*eplasequi*2e16)/(d_lattice*1e6) 
        elseif (eplasequi.gt.0.5) then
          xNt(1)=sqrt(2.d0)*(1e16)/(d_lattice*1e6)
          du2=0.d0
        endif  
	   xNt(2) = props(3)  ! M-A interfaces (sites/mm^3)
       xNt(3) = props(4)  ! Carbides (sites/mm^3)
       xNt(4) = props(5)  ! Grain Boundaries (sites/mm^3)
	   end if
	   ! Store number of trapps per unit lattice volume
	   statev(26)=xNt(1) ! Dislocations density (sites/mm^3)
	   statev(27)=Trans_peeq(locnum,npt)
	   
	   ! Solve thermal analogy equations
	   dudt2=0.d0
       do k1=1,ntraps
       dudt2=dudt2+xNt(k1)*xK(k1)*xNl/((xK(k1)*temp/conc_factor+xNl)**2.d0)
       end do
	   dudt=1.d0+dudt2
	   u=u+dudt*dtemp+du2+pen_D*((temp - C_env)*pen_fac*pen_wt)*dtime

		statev(28)=D
		do i=1,ntgrd
		dudg(i)=0.d0
		flux(i)=-D*dtemdx(i)+D*temp*Vh*sig(i)/(R*T)
		dfdt(i)=D*Vh*sig(i)/(R*T)
		dfdg(i,i)=-D
		end do
		  
		! Solve Lattice concentration
		cL = temp+dtemp
		statev(29)=cL ! Lattice concentration (wt ppm)
	    statev(30)=sig(1)
		statev(31)=sig(2)
		  
	  ! Solve Traps concentration
	  SumCT=0.d0
       do k1=1,ntraps
        xK(k1) = exp(Wb(k1)*1e6/(R*T))
        cT(k1) = xNt(k1) * xK(k1) * cL/conc_factor
     1 / ( xK(k1) * cL/conc_factor + xNl ); ![H atoms/mm^3]
        thetaT(k1) = cT(k1) / xNt(k1) !Trap occupancy [-]
        SumCT = SumCT + cT(k1) * ( 1.0d0 - thetaT(k1) )
      end do
	  ! Store results in SDV
      statev(32)=cT(1)*conc_factor !  Dislocations (wt ppm)
      statev(33)=cT(2)*conc_factor !  Martensitic-austenitic interfaces (wt ppm)
	  statev(34)=cT(3)*conc_factor !  Cementite-ferrite interfaces  (wt ppm)
	  statev(35)=cT(4)*conc_factor !  Grain Boundaries  (wt ppm)
	  statev(36)= cL +cT(1)*conc_factor+cT(2)*conc_factor +cT(3)*conc_factor 
     1 +cT(4)*conc_factor ! Total concentration (wt ppm)
	  statev(37)= statev(32)+statev(33)+statev(34)+statev(35) ! Ctraps (wt ppm)
	  ! Transfer Hydrogen Concentration
	  TranscL(locnum,npt)= temp+dtemp
      end if
	  
      end subroutine umatht



!***********************************************************************
! User defined subroutine for hardening.
!***********************************************************************
      subroutine ahard(syield,hard,eqplas,table,nvalue,jelem,coords,temp,emod0)
c
      use ktransfer
      include 'aba_param.inc'

      dimension table(2,nvalue) ! pointer for yield stress vs plastic strain
      dimension sy_vec(nvalue), ep_vec(nvalue) ! vectors

      ! get the vectors from the pointer 'table'
      sy_vec = table(1,1:nvalue) !yield stresses
      ep_vec = table(2,1:nvalue) !cum. eqv. plastic strain or N-value for power law hardening

      if (kflagHard.eq.0) call interp1(syield,hard,eqplas,sy_vec,ep_vec,nvalue)
      if (kflagHard.eq.1) call powerlawhardening(syield,hard,eqplas,sy_vec,ep_vec,nvalue,emod0)
      end subroutine ahard

!***********************************************************************
! Power law hardening: syield = Sy * (1 + alpha * E*ep/Sy)^N
!***********************************************************************
      subroutine powerlawhardening(syield,hard,eqplas,sy_vec,ep_vec,nvalue,E)

       use ktransfer
       include 'aba_param.inc'

       dimension sy_vec(nvalue), ep_vec(nvalue) ! vectors
       parameter (alpha = 1.d0) ! hardening parameter

       Sy = sy_vec(1) ! initial yield strength
       xn = ep_vec(1) ! hardening exponent
       syield = Sy*(1.d0+alpha*E*eqplas/Sy)**xn
       hard = xn*alpha*E*(1.d0+alpha*E*eqplas/Sy)**(xn-1.d0)
      end subroutine powerlawhardening


!***********************************************************************
!   Linear interpolation
!***********************************************************************
      subroutine interp1(y,dydx,x,y_vec,x_vec,nvalue)

      use ktransfer
      include 'aba_param.inc'

      dimension y_vec(nvalue), x_vec(nvalue) ! vectors

c    set yield stress to last value of table, hardening to zero
      y=y_vec(nvalue)
      dydx=0.d0

c   if more than one entry, search table
      if(nvalue.gt.1) then
        do 10 k1=1,nvalue-1
           x1=x_vec(k1+1)
           if(x.lt.x1) then ! find the range in x_vec 
             x0=x_vec(k1)
             if(x1.le.x0) then
                write(6,1)
 1              format('***error - x values must be entered in ascending order')
                call xit
              endif

c           current yield stress and hardening
            dx=x1-x0
            y0=y_vec(k1)
            y1=y_vec(k1+1)
            dy=y1-y0
            dydx=dy/dx
            y=y0+(x-x0)*dydx ! linear interpolation
            goto 20
            endif
 10         continue
 20         continue
      endif
      end subroutine interp1
!***********************************************************************
!    Key parameters for different PFMs
!***********************************************************************      
      subroutine phasefieldModel(kflagPFM, xlc, Gc, xi, c0, Hmin)

      use ktransfer, only: pi
      include 'aba_param.inc'

      if (kflagPFM.eq.0) then ! AT2
        xi=0.d0; c0=2.d0;
        Hmin=0.d0
      elseif (kflagPFM.eq.1) then ! AT1
        xi=1.d0; c0=8.d0/3.d0
        Hmin=Gc/(2.d0*xlc*c0)
      elseif (kflagPFM.eq.2) then ! PF-CZM
        xi=2.d0; c0=pi
        Hmin = max(Hmin, usr_tolerance)
      elseif (kflagPFM.eq.3) then
        xi=0.d0; c0=2.d0;
        Hmin = max(Hmin, usr_tolerance)
      else
        write(6,*)'Unknown flag for PFM'
        call xit
      endif
      end subroutine phasefieldModel


!***********************************************************************
!   Get phase field functions and their derivatives wrt phi
!***********************************************************************      
      subroutine phasefieldFunctions (kflagPFM, xlc, Gc, Hmin, phi, 
     1       alph, dalph, ddalph, omega, domega, ddomega)

      use ktransfer, only: pi
      include 'aba_param.inc'

      parameter (kappa = 1.d-9) ! residual strength

      ! Get the key parameters (xi, c0, Hmin)
      call phasefieldModel(kflagPFM, xlc, Gc, xi, c0, Hmin)

      ! Deometric function
      alph = xi * phi + (1.d0 - xi)*phi**2.d0
      dalph = xi + 2.d0*(1.d0 - xi)*phi
      ddalph = 2.d0*(1.d0-xi)

      ! Degradation function
      if ((kflagPFM.eq.0).or. (kflagPFM.eq.1)) then ! AT1/AT2
        omega = (1.d0-phi)**2.d0
        domega = -2.d0+2.d0*phi
        ddomega = 2.d0
      
      elseif (kflagPFM.eq.2) then ! PF-CZM
        xlch = 0.5d0 * Gc / Hmin; 
        a1 = 4.d0/c0*xlch/xlc;
        p  =  2.d0; a2 = -0.5d0; a3 =  0.0d0

        !if(a1.le.2.d0) write(6,*)'Small a1:', a1,' reduce l0 for stability'

        fac1    =  (1.d0 - phi)**p
        dfac1   = -p*(1.d0 - phi)**(p - 1.d0); 
        ddfac1  =  p*(p - 1.d0)*(1.d0 - phi)**(p - 2.d0)
        
        fac2    =  fac1   + a1*phi + a1*a2*phi**2.d0 + a1*a2*a3*phi**3.d0
        dfac2   =  dfac1  + a1 + 2.d0*a1*a2*phi + 3.d0*a1*a2*a3*phi**2.d0
        ddfac2  =  ddfac1 + 2.d0*a1*a2 + 6.d0*a1*a2*a3*phi
        
        omega   =  fac1/fac2        
        domega  =  (dfac1*fac2  - fac1*dfac2)/(fac2**2.d0)
        ddomega = ((ddfac1*fac2 - fac1*ddfac2)*fac2 - 2.d0*
     1             (dfac1*fac2 - fac1*dfac2)*dfac2)/(fac2**3.d0)
      
      elseif (kflagPFM.eq.3) then 
       b = 2.d0;
       xlch = 0.5d0 * Gc / Hmin; 
       a = (xlch/xlc);
       omega = (1 - exp(-a*(1 - phi)**b))/(1 - exp(-a))
       domega = a*b*(1 - phi)**(b - 1)*exp(a*(1 - (1 - phi)**b))/(1 - exp(a))
       ddomega = -a*b*(1 - phi)**(b - 2)*(a*b*(1 - phi)**b - b + 1)*exp(-a*(1 - phi)**b + a)/(exp(a) - 1)
      else
        write(6,*)'Unknown flag for PFM'
        call xit
      endif

      ! include a residual strength
      omega = omega*(1.d0-kappa) + kappa
      domega = domega*(1.d0-kappa)
      ddomega = ddomega*(1.d0-kappa)

      !write(*,*) 'PFM: Hmin, xlc, xlch, a1:: ', Hmin, xlc, xlch, a1
      !write(*,*) 'PFM: ', phi, omega, domega, ddomega

      end subroutine phasefieldFunctions


!***********************************************************************
!   Get value of degradation function of PFM (reqd. in umat)
!***********************************************************************      
      subroutine phasefieldDegFunc (kflagPFM, phi, xlc, Gc, Hmin, omega)

      use ktransfer, only: pi
      include 'aba_param.inc'

      parameter (kappa = 1.d-9) ! residual strength

      ! get the key parameters
      call phasefieldModel(kflagPFM, xlc, Gc, xi, c0, Hmin)

      ! degradation function
      if ((kflagPFM.eq.0).or. (kflagPFM.eq.1)) then ! AT1/AT2
        omega = (1.d0-phi)**2.d0
      
      elseif (kflagPFM.eq.2) then ! PF-CZM
        xlch = 0.5d0 * Gc / Hmin; 
        a1 = 4.d0/c0*xlch/xlc;
        p  =  2.d0; a2 = -0.5d0; a3 =  0.0d0

        fac1 =  (1.d0 - phi)**p
        fac2 =  fac1 + a1*phi + a1*a2*phi**2.d0 + a1*a2*a3*phi**3.d0

        omega =  fac1/fac2

      elseif (kflagPFM.eq.3) then 
       b = 2.d0;
       xlch = 0.5d0 * Gc / Hmin; 
       a = (xlch/xlc);
       omega = (1 - exp(-a*(1 - phi)**b))/(1 - exp(-a))

      else
        write(6,*)'Unknown flag for PFM'
        call xit
      endif

      ! Include a residual strength
      omega = omega*(1.d0-kappa) + kappa

      end subroutine phasefieldDegFunc
	  
